#!/bin/bash

set -e
set -u

export DEBIAN_FRONTEND=noninteractive

REPO="http://sbs2.konicaminolta.fr/epc"

DEFAULT_ENV="prod"

# Ensure we have the rights
test "$(whoami)" ==  "root" || (echo "You must be root to execute this script." ; exit 1)


if [ $# -eq 1 ] ; then
    if [ "${1}" == "prod" ] || [ "${1}" == "preprod" ] || [ "${1}" == "dev" ] ; then
        ENVIRONMENT="${1}"
    else
        echo "Bad environment. Must be 'dev', 'preprod' or 'prod'"
        exit 1
    fi
else
    ENVIRONMENT="${DEFAULT_ENV}"
fi

LOGDIR="/var/log/solimp"
mkdir -p "${LOGDIR}"
test -d "${LOGDIR}" || (echo "Could not create log directory" ; exit 1)
logfile="$LOGDIR/install.log"

#
# Log function
#
function log() {
    severity=${1}
    message=${2}
    echo "[$(date -R)] - [${severity}] : ${message}" >> "${logfile}"
}

#
# Log Command (execute command and log)
#
function logCommand() {
    command=${1}
    ($command | sed "s/^/[$(date -R)] - [INFO] : /" >> "${logfile}" ) 2>&1 | sed "s/^/[$(date -R)] - [INFO] : /" >> "${logfile}"
}

#
# Simple confirm command (default answer is NO)
#
function confirm () {
    local step=$1
    local confirm="n"
    local result=0
    echo -n "$step ? [y/N] "
    read -r confirm
    echo "${confirm}" | grep -q "^[Yy]$" || confirm="n"
    if [ "${confirm}" = "n" ] ; then
        result="1"
    fi
    return "${result}"
}

#
# Add normal user to system
#
function addUser () {
    id=$1
    user=$2
    pwd=$3
    homedir=$4
    [ ! -d "${homedir}" ] && mkdir -p "${homedir}"
    if ! getent passwd "${user}" >/dev/null ; then
        useradd -u "${id}" -m -s /bin/bash -d "${homedir}/${user}" "${user}" -p "${pwd}"
        chown -R "${user}" "${homedir}/${user}"
        log "INFO" "User ${user} added successfully."
    else
        log "INFO" "User ${user} already exists."
    fi
}

if ! grep -q "9." /etc/debian_version && \
   ! grep -q "10." /etc/debian_version && \
   ! grep -q "11." /etc/debian_version; then
    echo "Only DEBIAN stretch, buster or bullseye are supported"
    exit 1
fi

if grep -q "9." /etc/debian_version; then
    DEBIAN_VERSION=stretch
fi

if grep -q "10." /etc/debian_version; then
    DEBIAN_VERSION=buster
fi

if grep -q "11." /etc/debian_version; then
    DEBIAN_VERSION=bullseye
fi

cat > /etc/apt/sources.list.d/km.list << EOF
deb ${REPO}-${DEBIAN_VERSION}/${ENVIRONMENT}/debian/ ${DEBIAN_VERSION} non-free
EOF

log "INFO" "AddUser kmbsf"
addUser 1001 'kmbsf' 'cq1cmdp3uC' '/home'

log "INFO" "Adding apt repository GPG key"
wget -O - "${REPO}-${DEBIAN_VERSION}/${ENVIRONMENT}/debian/serians_apt.gpg.key" | apt-key add -

log "INFO" "apt update"
apt update

dostep=0
confirm "Running apt dist-upgrade (this may take some time)" || dostep=1
if [ $dostep -eq 0 ] ; then
    log "INFO" "apt dist-upgrade"
    apt dist-upgrade
fi

if ! grep -q "net.ifnames" /etc/default/grub ; then
    log "INFO" "Revert to ethX interfaces"
    sed -i -e 's/GRUB_CMDLINE_LINUX=""/GRUB_CMDLINE_LINUX="net.ifnames=0 biosdevname=0"/' /etc/default/grub

    log "INFO" "Generate GRUB config"
    grub-mkconfig -o /boot/grub/grub.cfg
fi

log "INFO" "Writing file /etc/network/interfaces"
cat > /etc/network/interfaces <<EOF
# Default interfaces
auto lo eth0 eth0:0
iface lo inet loopback

allow-hotplug eth0
iface eth0 inet dhcp

iface eth0:0 inet static
  address 192.168.168.1
  netmask 255.255.255.0
  broadcast 192.168.1.255

EOF

log "INFO" "Install mysql packages..."
apt install mariadb-server

log "INFO" "Changing timezone to Europe/Paris"
echo "Europe/Paris" > /etc/timezone
logCommand "dpkg-reconfigure -f noninteractive tzdata"

log "INFO" "Generating locales"
cp /etc/locale.gen /etc/locale.gen.dpkg
cat > /etc/locale.gen << EOF
en_GB.UTF-8 UTF-8
en_US.UTF-8 UTF-8
fr_FR ISO-8859-1
fr_FR.UTF-8 UTF-8
EOF

log "INFO" "Create database..."
mysql < user.sql
mysql -D kmbsf < epc.sql
mysql -D kmbsf < kmbsf_kmbsf_user.sql
mysql -D kmbsf < kmbsf_kmbsf_customerField.sql

logCommand "locale-gen"
#logCommand "update-locale LANG='fr_FR.UTF-8' LANGUAGE='fr_FR'"

log "INFO" "apt install solimp3-s1"
apt install solimp3-s1

log "INFO" "apt clean"
apt clean

log "INFO" "apt autoremove"
apt-get autoremove

confirm "Reboot ?" || dostep=1
if [ "${dostep}" -eq 0 ] ; then
    log "INFO" "Reboot"
    reboot
fi

exit 0
