#!/bin/bash

set -e
set -u

export DEBIAN_FRONTEND=noninteractive

REPO="http://apt.serians.fr/epc/"

DEFAULT_ENV="prod"

# Ensure we have the rights
test `whoami` ==  "root" || (echo "You must be root to execute this script." ; exit 1)


if [ $# -eq 1 ] ; then
    if [ $1 == "prod" ] || [ $1 == "preprod" ] || [ $1 == "dev" ] ; then
	ENVIRONMENT=$1
    else
	echo "Bad environment. Must be 'dev', 'preprod' or 'prod'"
	exit 1
    fi
else
    ENVIRONMENT=$DEFAULT_ENV
fi

LOGDIR="/var/log/solimp"
mkdir -p $LOGDIR
test -d $LOGDIR || (echo "Could not create log directory" ; exit 1)
logfile="$LOGDIR/install.log"

#
# Log function
#
function log() {
    severity=$1
    message=$2
    echo "[`date -R`] - [$severity] : $message" >> $logfile
}

#
# Log Command (execute command and log)
#
function logCommand() {
    command=$1
    ($command | sed "s/^/[`date -R`] - [INFO] : /" >> $logfile ) 2>&1 | sed "s/^/[`date -R`] - [INFO] : /" >> $logfile
}

#
# Simple confirm command (default answer is NO)
#
function confirm () {
    local step=$1
    local confirm="n"
    local result=0
    echo -n "$step ? [y/N] "
    read confirm
    echo $confirm | grep "^[Yy]$" 2>&1 1>/dev/null || confirm="n"
    if [ "$confirm" = "n" ] ; then
        result="1"
    fi
    return $result
}

DEB_VERSION=$(cat /etc/debian_version)
if echo $DEB_VERSION | grep -q "9." ; then
  CODENAME="stretch"
else
    echo "Only DEBIAN stretch is supported"
  exit 1
fi

cat > /etc/apt/sources.list.d/serians.list << EOF
deb $REPO/$ENVIRONMENT/debian/ stretch non-free
EOF

log "INFO" "Adding apt repository GPG key"
wget -O - "$REPO/$ENVIRONMENT/debian/serians_apt.gpg.key" | apt-key add -

log "INFO" "apt update"
apt update

dostep=0
confirm "Running apt dist-upgrade (this may take some time)" || dostep=1
if [ $dostep -eq 0 ] ; then
  log "INFO" "apt dist-upgrade"
  apt dist-upgrade
fi

log "INFO" "apt install sudo apache2"
apt install sudo apache2

if ! grep -q "net.ifnames" /etc/default/grub ; then
  log "INFO" "Revert to ethX interfaces"
  sed -i -e 's/GRUB_CMDLINE_LINUX=""/GRUB_CMDLINE_LINUX="net.ifnames=0 biosdevname=0"/' /etc/default/grub

  log "INFO" "Generate GRUB config"
  grub-mkconfig -o /boot/grub/grub.cfg
fi

log "INFO" "Writing file /etc/network/interfaces"
cat > /etc/network/interfaces <<EOF
# Default interfaces
auto lo eth0 eth0:0
iface lo inet loopback

allow-hotplug eth0
iface eth0 inet dhcp

iface eth0:0 inet static
  address 192.168.168.1
  netmask 255.255.255.0
  broadcast 192.168.1.255

EOF

#
# Suppression PermitRootLogin
#
#log "INFO" "SSH - Change listening port"
#sed -i -e 's/Port 22/Port 12020/' /etc/ssh/sshd_config

#log "INFO" "SSH - Disabling Root Login"
#sed -i -e 's/PermitRootLogin yes/PermitRootLogin no/' /etc/ssh/sshd_config

log "INFO" "Install mysql packages..."
apt install mysql-server

log "INFO" "Install web packages..."
apt install apache2 libapache2-mod-php libdbi-perl libclass-dbi-mysql-perl php7.0-mysql net-tools php7.0-mbstring

log "INFO" "Install snmp packages..."
apt install libnet-address-ip-local-perl libnet-snmp-perl libsnmp-perl libnet-ip-perl

log "INFO" "apt-get install libnet-ssleay-perl libmime-lite-perl libfile-libmagic-perl libdigest-hmac-perl libjson-perl libreadonly-perl"
apt install libnet-ssleay-perl libmime-lite-perl libfile-libmagic-perl libdigest-hmac-perl libjson-perl libreadonly-perl

log "INFO" "Changing timezone to Europe/Paris"
echo "Europe/Paris" > /etc/timezone
logCommand "dpkg-reconfigure -f noninteractive tzdata"

log "INFO" "Writing systemd apache2 service"
cat > /lib/systemd/system/apache2.service << EOF
[Unit]
Description=The Apache HTTP Server
After=network.target remote-fs.target nss-lookup.target

[Service]
Type=forking
Environment=APACHE_STARTED_BY_SYSTEMD=true
ExecStart=/usr/sbin/apachectl start
ExecStop=/usr/sbin/apachectl stop
ExecReload=/usr/sbin/apachectl graceful
PrivateTmp=false
Restart=on-abort

[Install]
WantedBy=multi-user.target
EOF

log "INFO" "Generating locales"
cp /etc/locale.gen /etc/locale.gen.dpkg
cat > /etc/locale.gen << EOF
en_GB.UTF-8 UTF-8
en_US.UTF-8 UTF-8
fr_FR ISO-8859-1
fr_FR.UTF-8 UTF-8
EOF

log "INFO" "Create database..."
mysql < user.sql
mysql -D kmbsf < epb.sql
mysql -D kmbsf < kmbsf_kmbsf_user.sql
mysql -D kmbsf < kmbsf_kmbsf_customerField.sql

logCommand "locale-gen"
#logCommand "update-locale LANG='fr_FR.UTF-8' LANGUAGE='fr_FR'"

apt install solimp3-s1

log "INFO" "apt clean"
apt clean

log "INFO" "apt autoremove"
apt-get autoremove

confirm "Reboot ?" || dostep=1
if [ $dostep -eq 0 ] ; then
  log "INFO" "Reboot"
  reboot
fi

exit 0
